<?php
// product_details.php
header('Content-Type: application/json; charset=utf-8');
// CORS (helpful for Expo / dev)
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

require_once __DIR__ . '/db.php';

// If you store relative media paths, prefix them here:
const UPLOAD_BASE = 'https://api.bayex.lk/api/uploads/';

function abs_url($u) {
    if (!$u) return null;
    if (preg_match('#^https?://#i', $u)) return $u;
    return rtrim(UPLOAD_BASE, '/').'/'.ltrim($u, '/');
}
function fetch_all($pdo, $sql, $params = []) {
    $st = $pdo->prepare($sql);
    $st->execute($params);
    return $st->fetchAll(PDO::FETCH_ASSOC);
}
function fetch_one($pdo, $sql, $params = []) {
    $st = $pdo->prepare($sql);
    $st->execute($params);
    return $st->fetch(PDO::FETCH_ASSOC);
}
function s($v) { return isset($v) ? (string)$v : null; }
function b($v) { return (int)(!!$v); }
function f($v, $d=0.0){ $x = (float)$v; return is_finite($x) ? $x : (float)$d; }

// ----------- Input -----------
$productId = isset($_GET['product_id']) ? (int)$_GET['product_id'] : 0;
if ($productId <= 0) {
    http_response_code(400);
    echo json_encode(['ok'=>false,'error'=>'Missing or invalid product_id']);
    exit;
}

try {
    // ----------- Product + Store -----------
    $product = fetch_one($pdo, "
        SELECT
            p.id, p.store_id, p.name, p.description,
            p.base_price, p.promo_price, p.has_variants, p.in_stock,
            p.rating_avg, p.rating_count, p.image_url, p.tags, p.is_active, p.created_at,
            s.name AS store_name, s.city AS store_city, s.logo_url AS store_logo_url
        FROM products p
        LEFT JOIN stores s ON s.id = p.store_id
        WHERE p.id = ?
        LIMIT 1
    ", [$productId]);

    if (!$product) {
        http_response_code(404);
        echo json_encode(['ok'=>false,'error'=>'Product not found']);
        exit;
    }

    $basePrice  = f($product['base_price'], 0);
    $promoPrice = f($product['promo_price'], 0);
    $hasPromo   = ($promoPrice > 0 && $promoPrice < $basePrice);
    $effPrice   = $hasPromo ? $promoPrice : $basePrice;
    $compareAt  = $hasPromo ? $basePrice : null;

    // ----------- Images -----------
    $images = [];
    if (!empty($product['image_url'])) {
        $images[] = ['url' => abs_url($product['image_url'])];
    }

    // ----------- Categories -----------
    $cats = fetch_all($pdo, "
        SELECT c.id, c.name, c.sort_order
        FROM product_category_map m
        INNER JOIN product_categories c ON c.id = m.category_id
        WHERE m.product_id = ?
        ORDER BY c.sort_order ASC, c.id ASC
    ", [$productId]);
    $categories = array_map(function($c){
        return ['id'=>(string)$c['id'], 'name'=>(string)$c['name']];
    }, $cats);

    // ----------- Variants (single-choice Option Group 'Variant') -----------
    $option_groups = [];
    if (b($product['has_variants'])) {
        $variants = fetch_all($pdo, "
            SELECT id, name, price, in_stock, sort_order
            FROM product_variants
            WHERE product_id = ?
            ORDER BY sort_order ASC, id ASC
        ", [$productId]);

        if ($variants) {
            $opts = array_map(function($v) use ($effPrice){
                $price = f($v['price'], 0);
                $delta = $price - $effPrice; // price_delta relative to effective base price
                return [
                    'id'          => (string)$v['id'],
                    'name'        => (string)$v['name'],
                    'price_delta' => (string)$delta,
                    'in_stock'    => (string)b($v['in_stock']),
                ];
            }, $variants);

            $option_groups[] = [
                'id'       => 'variant',
                'name'     => 'Variant',
                'type'     => 'single',   // radio/single-choice
                'required' => true,
                'options'  => $opts,
            ];
        }
    }

    // ----------- Modifiers -> Option/Addon groups -----------
    // Rule:
    // - max_select = 1  -> option_groups (single-choice)
    // - max_select != 1 -> addon_groups   (multi-select)
    $groups = fetch_all($pdo, "
        SELECT id, name, min_select, max_select, required, sort_order
        FROM modifier_groups
        WHERE product_id = ?
        ORDER BY sort_order ASC, id ASC
    ", [$productId]);

    $addon_groups = [];
    foreach ($groups as $g) {
        $mods = fetch_all($pdo, "
            SELECT id, name, price, sort_order
            FROM modifiers
            WHERE group_id = ?
            ORDER BY sort_order ASC, id ASC
        ", [$g['id']]);

        $mapped = array_map(function($m){
            return [
                'id'          => (string)$m['id'],
                'name'        => (string)$m['name'],
                // Modifiers add on top of base/variant → treat as delta
                'price_delta' => (string)f($m['price'], 0),
            ];
        }, $mods);

        $max = isset($g['max_select']) ? (int)$g['max_select'] : null;

        if ($max === 1) {
            // Single-choice → option group
            $option_groups[] = [
                'id'       => (string)$g['id'],
                'name'     => (string)$g['name'],
                'type'     => 'single',
                'required' => (bool)$g['required'],
                'options'  => $mapped,
            ];
        } else {
            // Multi-select → addon group
            $addon_groups[] = [
                'id'    => (string)$g['id'],
                'name'  => (string)$g['name'],
                'min'   => is_null($g['min_select']) ? null : (int)$g['min_select'],
                'max'   => is_null($g['max_select']) ? null : (int)$g['max_select'],
                'options' => $mapped,
            ];
        }
    }

    // ----------- Tags (optional) -----------
    $tags = [];
    if (!empty($product['tags'])) {
        foreach (explode(',', (string)$product['tags']) as $t) {
            $t = trim($t);
            if ($t !== '') $tags[] = $t;
        }
    }

    // ----------- Build response -----------
    $resp = [
        'ok' => true,
        'product' => [
            'id'               => (string)$product['id'],
            'name'             => (string)$product['name'],
            'description'      => s($product['description']) ?? '',
            'price'            => (string)$effPrice,
            'compare_at_price' => isset($compareAt) ? (string)$compareAt : null,
            'in_stock'         => (string)b($product['in_stock']),
            'min_qty'          => '1',
            'max_qty'          => '99',
            'image_url'        => abs_url($product['image_url']),
            'images'           => $images,  // gallery fallback is image_url only
            'rating_avg'       => s($product['rating_avg']) ?? '0',
            'rating_count'     => s($product['rating_count']) ?? '0',
            'tags'             => $tags,
            'categories'       => $categories,
            'store' => [
                'id'       => (string)$product['store_id'],
                'name'     => s($product['store_name']) ?? '',
                'city'     => s($product['store_city']) ?? '',
                'logo_url' => abs_url($product['store_logo_url'] ?? ''),
            ],
            'option_groups'    => $option_groups,
            'addon_groups'     => $addon_groups,
        ]
    ];

    echo json_encode($resp);
    exit;

} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode([
        'ok' => false,
        'error' => 'Server error',
        // In production you may want to hide the detail:
        'detail' => $e->getMessage()
    ]);
    exit;
}
